using System;
using System.Collections.Generic;

namespace UnityEngine.Experimental.Rendering.RenderGraphModule
{
    /// <summary>
    /// Use this struct to set up a new Render Pass.
    /// </summary>
    public struct RenderGraphBuilder : IDisposable
    {
        RenderGraph.RenderPass      m_RenderPass;
        RenderGraphResourceRegistry m_Resources;
        bool                        m_Disposed;

        #region Public Interface
        /// <summary>
        /// Specify that the pass will use a Texture resource as a color render target.
        /// This has the same effect as WriteTexture and also automatically sets the Texture to use as a render target.
        /// </summary>
        /// <param name="input">The Texture resource to use as a color render target.</param>
        /// <param name="index">Index for multiple render target usage.</param>
        /// <returns>An updated resource handle to the input resource.</returns>
        public RenderGraphMutableResource UseColorBuffer(in RenderGraphMutableResource input, int index)
        {
            if (input.type != RenderGraphResourceType.Texture)
                throw new ArgumentException("Trying to write to a resource that is not a texture or is invalid.");

            m_RenderPass.SetColorBuffer(input, index);
            m_Resources.UpdateTextureFirstWrite(input, m_RenderPass.index);
            return input;
        }

        /// <summary>
        /// Specify that the pass will use a Texture resource as a depth buffer.
        /// </summary>
        /// <param name="input">The Texture resource to use as a depth buffer during the pass.</param>
        /// <param name="flags">Specify the access level for the depth buffer. This allows you to say whether you will read from or write to the depth buffer, or do both.</param>
        /// <returns>An updated resource handle to the input resource.</returns>
        public RenderGraphMutableResource UseDepthBuffer(in RenderGraphMutableResource input, DepthAccess flags)
        {
            if (input.type != RenderGraphResourceType.Texture)
                throw new ArgumentException("Trying to write to a resource that is not a texture or is invalid.");

            m_RenderPass.SetDepthBuffer(input, flags);
            if ((flags | DepthAccess.Read) != 0)
                m_Resources.UpdateTextureLastRead(input, m_RenderPass.index);
            if ((flags | DepthAccess.Write) != 0)
                m_Resources.UpdateTextureFirstWrite(input, m_RenderPass.index);
            return input;
        }

        /// <summary>
        /// Specify a Texture resource to read from during the pass.
        /// </summary>
        /// <param name="input">The Texture resource to read from during the pass.</param>
        /// <returns>An updated resource handle to the input resource.</returns>
        public RenderGraphResource ReadTexture(in RenderGraphResource input)
        {
            if (input.type != RenderGraphResourceType.Texture)
                throw new ArgumentException("Trying to read a resource that is not a texture or is invalid.");
            m_RenderPass.resourceReadList.Add(input);
            m_Resources.UpdateTextureLastRead(input, m_RenderPass.index);
            return input;
        }

        /// <summary>
        /// Specify a Texture resource to write to during the pass.
        /// </summary>
        /// <param name="input">The Texture resource to write to during the pass.</param>
        /// <returns>An updated resource handle to the input resource.</returns>
        public RenderGraphMutableResource WriteTexture(in RenderGraphMutableResource input)
        {
            if (input.type != RenderGraphResourceType.Texture)
                throw new ArgumentException("Trying to write to a resource that is not a texture or is invalid.");
            // TODO: Manage resource "version" for debugging purpose
            m_RenderPass.resourceWriteList.Add(input);
            m_Resources.UpdateTextureFirstWrite(input, m_RenderPass.index);
            return input;
        }

        /// <summary>
        /// Specify a Renderer List resource to use during the pass.
        /// </summary>
        /// <param name="input">The Renderer List resource to use during the pass.</param>
        /// <returns>An updated resource handle to the input resource.</returns>
        public RenderGraphResource UseRendererList(in RenderGraphResource input)
        {
            if (input.type != RenderGraphResourceType.RendererList)
                throw new ArgumentException("Trying use a resource that is not a renderer list.");
            m_RenderPass.usedRendererListList.Add(input);
            return input;
        }

        /// <summary>
        /// Specify the render function to use for this pass.
        /// A call to this is mandatory for the pass to be valid.
        /// </summary>
        /// <typeparam name="PassData">The Type of the class that provides data to the Render Pass.</typeparam>
        /// <param name="renderFunc">Render function for the pass.</param>
        public void SetRenderFunc<PassData>(RenderFunc<PassData> renderFunc) where PassData : class, new()
        {
            ((RenderGraph.RenderPass<PassData>)m_RenderPass).renderFunc = renderFunc;
        }

        /// <summary>
        /// Enable asynchronous compute for this pass.
        /// </summary>
        /// <param name="value">Set to true to enable asynchronous compute.</param>
        public void EnableAsyncCompute(bool value)
        {
            m_RenderPass.enableAsyncCompute = value;
        }

        /// <summary>
        /// Dispose the RenderGraphBuilder instance.
        /// </summary>
        public void Dispose()
        {
            Dispose(true);
        }
        #endregion

        #region Internal Interface
        internal RenderGraphBuilder(RenderGraph.RenderPass renderPass, RenderGraphResourceRegistry resources)
        {
            m_RenderPass = renderPass;
            m_Resources = resources;
            m_Disposed = false;
        }

        void Dispose(bool disposing)
        {
            if (m_Disposed)
                return;

            m_Disposed = true;
        }
        #endregion
    }
}
